#region Using declarations
using System;
using System.ComponentModel.DataAnnotations;
using NinjaTrader.Cbi;
using NinjaTrader.Gui.Tools;
using NinjaTrader.Data;
using NinjaTrader.NinjaScript;
using NinjaTrader.NinjaScript.Indicators;
#endregion
namespace NinjaTrader.NinjaScript.Strategies.NInjaProTrader_CursoProgramacao
{
    public class RompimentoEMA : Strategy
    {
        // ----- Indicadores -----
        private EMA emaSetup;
        private EMA emaTrail;
        // ----- Controle diário / PnL -----
        private DateTime dataAtual = Core.Globals.MinDate;
        private double startingCumProfit = 0.0;
        private double lastCumProfit = 0.0;
        // ----- Controle do trade atual (trailing) -----
        private bool trailingAtivo = false;
        private bool alvoDesativadoNoTrailing = false; // NOVO: alvo some quando trailing ativa
        private double precoEntrada = 0.0;
        private double alvoVirtual = 0.0; // alvo "virtual" para medir % de avanço
        private double stopAtual = 0.0; // espelha o SetStopLoss atual
        private double tamanhoBarraEntrada = 0.0; // range da barra do setup
        private double maxPrecoDesdeEntrada = 0.0; // para long
        private double minPrecoDesdeEntrada = double.MaxValue; // para short
        // ---------------- PARÂMETROS ----------------
        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "Período da EMA (Setup)", Order = 1, GroupName = "Parâmetros")]
        public int PeriodoEMA { get; set; }
        [NinjaScriptProperty]
        [Range(0.0, 1.0)]
        [Display(Name = "Percentual mínimo do corpo (0-1)", Order = 3, GroupName = "Parâmetros")]
        public double CorpoMinimoPercentual { get; set; }
        [NinjaScriptProperty]
        [Range(0.0001, double.MaxValue)]
        [Display(Name = "Tamanho máximo da barra (pontos)", Order = 4, GroupName = "Parâmetros")]
        public double TamanhoMaximoBarra { get; set; }
        // --------- GERENCIAMENTO DE RISCO ----------
        [NinjaScriptProperty]
        [Display(Name = "Ativar limite financeiro por operação?", Order = 5, GroupName = "Gerenciamento de Risco")]
        public bool AtivarLimiteFinanceiro { get; set; }
        [NinjaScriptProperty]
        [Range(0.01, double.MaxValue)]
        [Display(Name = "Limite financeiro por operação (USD)", Order = 6, GroupName = "Gerenciamento de Risco")]
        public double LimiteFinanceiro { get; set; }
        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "Quantidade de contratos", Order = 7, GroupName = "Gerenciamento de Risco")]
        public int QuantidadeContratos { get; set; }
        [NinjaScriptProperty]
        [Range(0.1, double.MaxValue)]
        [Display(Name = "Multiplicador do Alvo", Order = 8, GroupName = "Gerenciamento de Risco")]
        public double MultiplicadorAlvo { get; set; }
        // --------- CONTROLE DIÁRIO ----------
        [NinjaScriptProperty]
        [Display(Name = "Ativar perda máxima diária?", Order = 8, GroupName = "Controle Diário")]
        public bool AtivarPerdaMaximaDiaria { get; set; }
        [NinjaScriptProperty]
        [Range(0.01, double.MaxValue)]
        [Display(Name = "Perda máxima diária (USD)", Order = 9, GroupName = "Controle Diário")]
        public double PerdaMaximaDiariaUSD { get; set; }
        [NinjaScriptProperty]
        [Display(Name = "Ativar ganho máximo diário?", Order = 10, GroupName = "Controle Diário")]
        public bool AtivarGanhoMaximoDiario { get; set; }
        [NinjaScriptProperty]
        [Range(0.01, double.MaxValue)]
        [Display(Name = "Ganho máximo diário (USD)", Order = 11, GroupName = "Controle Diário")]
        public double GanhoMaximoDiarioUSD { get; set; }
        // --------- LOGS ----------
        [NinjaScriptProperty]
        [Display(Name = "Ativar log de operações?", Order = 12, GroupName = "Logs")]
        public bool AtivarLogOperacoes { get; set; }
        [NinjaScriptProperty]
        [Display(Name = "Ativar log de bloqueios?", Order = 13, GroupName = "Logs")]
        public bool AtivarLogBloqueios { get; set; }
        // --------- JANELA HORÁRIA ----------
        [NinjaScriptProperty]
        [Display(Name = "Ativar janela de horário?", Order = 14, GroupName = "Horário")]
        public bool AtivarJanelaHorario { get; set; }
        [NinjaScriptProperty]
        [Range(0, 2359)]
        [Display(Name = "Horário início (HHmm)", Order = 15, GroupName = "Horário")]
        public int HorarioInicio { get; set; }
        [NinjaScriptProperty]
        [Range(0, 2359)]
        [Display(Name = "Horário fim (HHmm)", Order = 16, GroupName = "Horário")]
        public int HorarioFim { get; set; }
        // --------- TRAILING POR EMA ----------
        [NinjaScriptProperty]
        [Display(Name = "Ativar trailing por EMA?", Order = 1, GroupName = "Trailing EMA")]
        public bool AtivarTrailingEMA { get; set; }
        [NinjaScriptProperty]
        [Range(1, int.MaxValue)]
        [Display(Name = "Período da EMA do Trailing", Order = 2, GroupName = "Trailing EMA")]
        public int PeriodoEMATrailing { get; set; }
        [NinjaScriptProperty]
        [Range(0.0, 1.0)]
        [Display(Name = "Percentual para ativar trailing (0-1)", Order = 3, GroupName = "Trailing EMA")]
        public double PercentualAtivarTrailing { get; set; }
        // --------- REVERSÃO ----------
        [NinjaScriptProperty]
        [Display(Name = "Permitir inverter posição?", Order = 17, GroupName = "Parâmetros")]
        public bool PermitirInverterPosicao { get; set; }
        // ======================================================
        protected override void OnStateChange()
        {
            if (State == State.SetDefaults)
            {
                Description = @"Rompimento da EMA com filtros, risco, janela de horário, logs, trailing por EMA e reversão opcional. Alvo é removido automaticamente quando o trailing ativa.";
                Name = "RompimentoEMA";
                Calculate = Calculate.OnBarClose;
                EntriesPerDirection = 1;
                EntryHandling = EntryHandling.AllEntries;
                IsExitOnSessionCloseStrategy = true;
                ExitOnSessionCloseSeconds = 30;
                BarsRequiredToTrade = 20;
                // Defaults
                PeriodoEMA = 21; // filtro
                MultiplicadorAlvo = 2.0;
                CorpoMinimoPercentual = 0.5;
                TamanhoMaximoBarra = 50.0;
                AtivarLimiteFinanceiro = true;
                LimiteFinanceiro = 100.0;
                QuantidadeContratos = 1;
                AtivarPerdaMaximaDiaria = true;
                PerdaMaximaDiariaUSD = 200.0;
                AtivarGanhoMaximoDiario = true;
                GanhoMaximoDiarioUSD = 700.0;
                AtivarLogOperacoes = true;
                AtivarLogBloqueios = true;
                AtivarJanelaHorario = true;
                HorarioInicio = 930;
                HorarioFim = 1600;
                // Trailing
                AtivarTrailingEMA = false;
                PeriodoEMATrailing = 8; // trailing pela EMA(8)
                PercentualAtivarTrailing = 0.80; // 80%
                // Reversão
                PermitirInverterPosicao = false; // padrão: só entra em Flat
            }
            else if (State == State.DataLoaded)
            {
                emaSetup = EMA(PeriodoEMA);
                emaTrail = EMA(PeriodoEMATrailing);
            }
        }
        private double GetCumulativeProfitSafe()
        {
            try { return SystemPerformance.AllTrades.TradesPerformance.Currency.CumProfit; }
            catch { return 0.0; }
        }
        private void ResetControleTrade()
        {
            trailingAtivo = false;
            alvoDesativadoNoTrailing = false;
            precoEntrada = 0.0;
            alvoVirtual = 0.0;
            stopAtual = 0.0;
            // NÃO zeramos tamanhoBarraEntrada aqui!
            maxPrecoDesdeEntrada = 0.0;
            minPrecoDesdeEntrada = double.MaxValue;
        }
        // ======================================================
        protected override void OnBarUpdate()
        {
            if (CurrentBar < BarsRequiredToTrade) return;
            // ---- rollover diário p/ metas/limites ----
            DateTime barraData = Time[0].Date;
            if (barraData != dataAtual.Date)
            {
                dataAtual = barraData;
                startingCumProfit = GetCumulativeProfitSafe();
                lastCumProfit = startingCumProfit;
            }
            double currentCumProfit = GetCumulativeProfitSafe();
            double lucroDia = currentCumProfit - startingCumProfit;
            if (AtivarPerdaMaximaDiaria && lucroDia <= -PerdaMaximaDiariaUSD)
            {
                if (AtivarLogBloqueios) Print(Time[0] + " Bloqueio perda diária.");
                GerenciarTrailingEMA();
                return;
            }
            if (AtivarGanhoMaximoDiario && lucroDia >= GanhoMaximoDiarioUSD)
            {
                if (AtivarLogBloqueios) Print(Time[0] + " Bloqueio ganho diário.");
                GerenciarTrailingEMA();
                return;
            }
            // ---- Janela de horário (HHmmss) ----
            if (AtivarJanelaHorario)
            {
                int horaAtual = ToTime(Time[0]); // HHmmss
                int horaInicio = HorarioInicio * 100; // 930 -> 093000
                int horaFim = HorarioFim * 100; // 1600 -> 160000
                if (horaAtual < horaInicio || horaAtual > horaFim)
                {
                    if (AtivarLogBloqueios) Print(Time[0] + " Fora da janela de horário. Nenhuma nova entrada.");
                    GerenciarTrailingEMA();
                    return;
                }
            }
            // ---- Gestão de trailing se já estiver posicionado ----
            GerenciarTrailingEMA();
            // ----------------- Filtros de setup -----------------
            double tamanhoBarra = High[0] - Low[0];
            double corpo = Math.Abs(Close[0] - Open[0]);
            if (tamanhoBarra > TamanhoMaximoBarra) return;
            if (corpo < (tamanhoBarra * CorpoMinimoPercentual)) return;
            // ======== CONDIÇÕES DE ENTRADA ========
            bool sinalCompra = Close[0] > Open[0] && Open[0] < emaSetup[0] && Close[0] > emaSetup[0];
            bool sinalVenda = Close[0] < Open[0] && Open[0] > emaSetup[0] && Close[0] < emaSetup[0];
            // ----- COMPRA -----
            if (sinalCompra && (Position.MarketPosition == MarketPosition.Flat ||
                                (PermitirInverterPosicao && Position.MarketPosition == MarketPosition.Short)))
            {
                ResetControleTrade();
                tamanhoBarraEntrada = High[0] - Low[0];
                double entry = Close[0];
                double stop = Low[0]; // mínima exata
                double riscoUSD = (entry - stop) * Instrument.MasterInstrument.PointValue * QuantidadeContratos;
                if (AtivarLimiteFinanceiro && riscoUSD > LimiteFinanceiro)
                {
                    if (AtivarLogBloqueios) Print(Time[0] + $" Bloqueio por risco: {riscoUSD:F2} > {LimiteFinanceiro:F2}");
                    return;
                }
                // NOVO: Verifica se a perda potencial desta operação excederia o limite diário
                double perdaPotencial = lucroDia - riscoUSD;
                if (AtivarPerdaMaximaDiaria && perdaPotencial <= -PerdaMaximaDiariaUSD)
                {
                    if (AtivarLogBloqueios) Print(Time[0] + $" Bloqueio por perda potencial diária: {perdaPotencial:F2} <= {-PerdaMaximaDiariaUSD:F2}");
                    return;
                }
                double alvo = entry + (tamanhoBarraEntrada * MultiplicadorAlvo);
                int qty = QuantidadeContratos + (PermitirInverterPosicao && Position.MarketPosition == MarketPosition.Short
                                                  ? Position.Quantity : 0);
                SetStopLoss("CompraEMA", CalculationMode.Price, stop, false);
                stopAtual = stop;
                // Alvo ATIVO na entrada
                SetProfitTarget("CompraEMA", CalculationMode.Price, alvo);
                EnterLong(qty, "CompraEMA");
            }
            // ----- VENDA -----
            if (sinalVenda && (Position.MarketPosition == MarketPosition.Flat ||
                               (PermitirInverterPosicao && Position.MarketPosition == MarketPosition.Long)))
            {
                ResetControleTrade();
                tamanhoBarraEntrada = High[0] - Low[0];
                double entry = Close[0];
                double stop = High[0]; // máxima exata
                double riscoUSD = (stop - entry) * Instrument.MasterInstrument.PointValue * QuantidadeContratos;
                if (AtivarLimiteFinanceiro && riscoUSD > LimiteFinanceiro)
                {
                    if (AtivarLogBloqueios) Print(Time[0] + $" Bloqueio por risco: {riscoUSD:F2} > {LimiteFinanceiro:F2}");
                    return;
                }
                // NOVO: Verifica se a perda potencial desta operação excederia o limite diário
                double perdaPotencial = lucroDia - riscoUSD;
                if (AtivarPerdaMaximaDiaria && perdaPotencial <= -PerdaMaximaDiariaUSD)
                {
                    if (AtivarLogBloqueios) Print(Time[0] + $" Bloqueio por perda potencial diária: {perdaPotencial:F2} <= {-PerdaMaximaDiariaUSD:F2}");
                    return;
                }
                double alvo = entry - (tamanhoBarraEntrada * MultiplicadorAlvo);
                int qty = QuantidadeContratos + (PermitirInverterPosicao && Position.MarketPosition == MarketPosition.Long
                                                  ? Position.Quantity : 0);
                SetStopLoss("VendaEMA", CalculationMode.Price, stop, false);
                stopAtual = stop;
                // Alvo ATIVO na entrada
                SetProfitTarget("VendaEMA", CalculationMode.Price, alvo);
                EnterShort(qty, "VendaEMA");
            }
        }
        // ======================================================
        private void GerenciarTrailingEMA()
        {
            if (!AtivarTrailingEMA) return;
            if (precoEntrada == 0) return; // ainda não houve entrada
            double distAlvoTotal = Math.Abs(alvoVirtual - precoEntrada);
            if (distAlvoTotal < 4 * TickSize) return; // evita alvo virtual ridículo
            if (Position.MarketPosition == MarketPosition.Long)
            {
                maxPrecoDesdeEntrada = Math.Max(maxPrecoDesdeEntrada, High[0]);
                if (!trailingAtivo)
                {
                    double progresso = (maxPrecoDesdeEntrada - precoEntrada) / distAlvoTotal;
                    if (progresso >= PercentualAtivarTrailing)
                    {
                        trailingAtivo = true;
                        DesativarAlvoAtual(); // <- remove alvo ao ativar trailing
                    }
                }
                if (trailingAtivo)
                {
                    double emaStop = emaTrail[0];
                    double candidato = Math.Max(stopAtual, emaStop);
                    candidato = Math.Min(candidato, Close[0] - TickSize);
                    candidato = Instrument.MasterInstrument.RoundToTickSize(candidato);
                    if (candidato > stopAtual)
                    {
                        SetStopLoss("CompraEMA", CalculationMode.Price, candidato, false);
                        stopAtual = candidato;
                        if (AtivarLogOperacoes) Print(Time[0] + $" [Trailing LONG] novo SL = {stopAtual:F2} (EMA{PeriodoEMATrailing}={emaTrail[0]:F2})");
                    }
                }
            }
            else if (Position.MarketPosition == MarketPosition.Short)
            {
                minPrecoDesdeEntrada = Math.Min(minPrecoDesdeEntrada, Low[0]);
                if (!trailingAtivo)
                {
                    double progresso = (precoEntrada - minPrecoDesdeEntrada) / distAlvoTotal;
                    if (progresso >= PercentualAtivarTrailing)
                    {
                        trailingAtivo = true;
                        DesativarAlvoAtual(); // <- remove alvo ao ativar trailing
                    }
                }
                if (trailingAtivo)
                {
                    double emaStop = emaTrail[0];
                    double candidato = Math.Min(stopAtual, emaStop);
                    candidato = Math.Max(candidato, Close[0] + TickSize);
                    candidato = Instrument.MasterInstrument.RoundToTickSize(candidato);
                    if (candidato < stopAtual)
                    {
                        SetStopLoss("VendaEMA", CalculationMode.Price, candidato, false);
                        stopAtual = candidato;
                        if (AtivarLogOperacoes) Print(Time[0] + $" [Trailing SHORT] novo SL = {stopAtual:F2} (EMA{PeriodoEMATrailing}={emaTrail[0]:F2})");
                    }
                }
            }
            else
            {
                ResetControleTrade();
            }
        }
        // Desativa o alvo atual quando o trailing ativa:
        // Como a API SetProfitTarget não possui "cancelar" explícito no modo Managed,
        // deslocamos o alvo para MUITO longe, tornando-o inalcançável (na prática, removido).
        private void DesativarAlvoAtual()
        {
            if (alvoDesativadoNoTrailing) return;
            double muitoLonge = 100000 * TickSize; // 100k ticks
            if (Position.MarketPosition == MarketPosition.Long)
            {
                double novoAlvo = Instrument.MasterInstrument.RoundToTickSize(Close[0] + muitoLonge);
                SetProfitTarget("CompraEMA", CalculationMode.Price, novoAlvo);
                if (AtivarLogOperacoes) Print(Time[0] + " [Trailing] Alvo desativado (LONG).");
            }
            else if (Position.MarketPosition == MarketPosition.Short)
            {
                double novoAlvo = Instrument.MasterInstrument.RoundToTickSize(Close[0] - muitoLonge);
                SetProfitTarget("VendaEMA", CalculationMode.Price, novoAlvo);
                if (AtivarLogOperacoes) Print(Time[0] + " [Trailing] Alvo desativado (SHORT).");
            }
            alvoDesativadoNoTrailing = true;
        }
        // ======================================================
        protected override void OnExecutionUpdate(Execution execution, string executionId, double price, int quantity,
            MarketPosition marketPosition, string orderId, DateTime time)
        {
            if (execution == null || execution.Order == null) return;
            if (execution.Order.OrderState != OrderState.Filled) return;
            if (execution.Order.Name == "CompraEMA" && marketPosition == MarketPosition.Long)
            {
                precoEntrada = price;
                maxPrecoDesdeEntrada = High[0];
                minPrecoDesdeEntrada = double.MaxValue;
                double alvoDist = Math.Max(tamanhoBarraEntrada * MultiplicadorAlvo, TickSize);
                alvoVirtual = precoEntrada + alvoDist;
                trailingAtivo = false;
                alvoDesativadoNoTrailing = false; // alvo volta a ser "normal" nesta nova posição
            }
            else if (execution.Order.Name == "VendaEMA" && marketPosition == MarketPosition.Short)
            {
                precoEntrada = price;
                minPrecoDesdeEntrada = Low[0];
                maxPrecoDesdeEntrada = 0.0;
                double alvoDist = Math.Max(tamanhoBarraEntrada * MultiplicadorAlvo, TickSize);
                alvoVirtual = precoEntrada - alvoDist;
                trailingAtivo = false;
                alvoDesativadoNoTrailing = false;
            }
            double currentCumProfit = GetCumulativeProfitSafe();
            double tradeProfit = currentCumProfit - lastCumProfit;
            lastCumProfit = currentCumProfit;
            if (AtivarLogOperacoes && Math.Abs(tradeProfit) > 0.0001)
                Print(time + " Trade realizado: " + tradeProfit.ToString("C2"));
        }
    }
}